"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SearchGuard = void 0;
var _routes = require("./routes");
var _auth = require("./auth");
var _routes2 = require("./system/routes");
var _routes3 = require("./configuration/routes/routes");
var _sanity_checks = require("./sanity_checks");
var _ReadOnlyMode = require("./authorization/ReadOnlyMode");
var _AuthManager = require("./auth/AuthManager");
var _routes_auth = require("./auth/routes_auth");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _statehood = _interopRequireDefault(require("@hapi/statehood"));
var _CustomCookieWrapper = require("./session/CustomCookieWrapper");
/* eslint-disable @kbn/eslint/require-license-header */

class SearchGuard {
  constructor(coreContext) {
    this.coreContext = coreContext;
    this.logger = this.coreContext.logger.get('searchguard');
  }
  async setup({
    core,
    pluginDependencies,
    configService,
    kibanaRouter,
    searchGuardBackend,
    searchGuardConfigurationBackend,
    spacesService,
    elasticsearch
  }) {
    this.logger.debug('Setup app');
    const kibanaVersionIndex = configService.get('kibana.index') + '_' + this.coreContext.env.packageInfo.version;

    //https://www.elastic.co/guide/en/elasticsearch/client/javascript-api/current/changelog-client.html#_move_from_emitter_like_interface_to_a_diagnostic_method
    elasticsearch.client.rootScopedClient.diagnostic.on('request', (0, _auth.rootScopedClientRequestWrapper)({
      configService,
      kibanaVersionIndex
    }));
    try {
      // Sanity checks
      (0, _sanity_checks.checkTLSConfig)({
        configService,
        logger: this.logger
      });
      (0, _sanity_checks.checkCookieConfig)({
        configService,
        logger: this.logger
      });
      (0, _sanity_checks.checkDoNotFailOnForbidden)({
        searchGuardBackend,
        logger: this.logger
      });

      // Inits the authInfo route
      (0, _auth.defineAuthInfoRoutes)({
        searchGuardBackend,
        kibanaCore: core,
        logger: this.logger
      });
      (0, _routes.registerRoutes)({
        router: kibanaRouter,
        config: configService.getConfig(),
        logger: this.logger
      });

      // Set up our own instance of Statehood
      const cookieOptions = (0, _CustomCookieWrapper.getSessionCookieOptions)(configService, core.http.basePath.get());
      const statehoodDefinitions = new _statehood.default.Definitions(cookieOptions);
      const sessionCookieName = configService.get('searchguard.cookie.name');
      statehoodDefinitions.add(sessionCookieName, cookieOptions);
      const sessionStorageFactory = {
        asScoped(request) {
          return new _CustomCookieWrapper.CustomCookieWrapper(statehoodDefinitions, (0, _coreHttpRouterServerInternal.ensureRawRequest)(request), sessionCookieName, cookieOptions);
        }
      };
      const authType = configService.get('searchguard.auth.type', null);
      let authManager = null;
      let kerberos = null;

      // Handle Kerberos separately because we don't want to bring up entire jungle from AuthType here.
      if (authType === 'kerberos') {
        kerberos = new _auth.Kerberos({
          pluginDependencies,
          config: configService,
          searchGuardBackend,
          logger: this.coreContext.logger.get('searchguard-kerberos-auth'),
          basePath: core.http.basePath.get(),
          sessionStorageFactory: sessionStorageFactory
        });
        core.http.registerOnPreAuth(kerberos.checkAuth);
      } else if (authType !== 'proxy') {
        authManager = new _AuthManager.AuthManager({
          kibanaCore: core,
          sessionStorageFactory,
          pluginDependencies,
          logger: this.coreContext.logger.get('searchguard-auth'),
          searchGuardBackend,
          configService,
          spacesService
        });
        authManager.registerAuthInstances();
        (0, _routes_auth.defineAuthRoutes)({
          kibanaCore: core,
          authManager,
          searchGuardBackend,
          configService
        });
        if (authManager) {
          // authManager.onPreAuth needs to run before any other handler
          // that manipulates the request headers (e.g. MT)
          core.http.registerOnPreAuth(authManager.onPreAuth);
          core.http.registerOnPreAuth(authManager.checkAuth);
          core.http.registerOnPostAuth(authManager.handleAuthForOptionalRoutes);
          // @todo Not really needed anymore after taking optional auth into account.
          //core.http.registerOnPostAuth(authManager.onPostAuth);
        }
      }
      if (authType && ['proxy', 'kerberos'].indexOf(authType) === -1) {
        try {
          this.logger.info('Initialising Search Guard authentication plugin.');
          if (configService.get('searchguard.cookie.password') === 'searchguard_cookie_default_password') {
            this.logger.warn("Default cookie password detected, please set a password in kibana.yml by setting 'searchguard.cookie.password' (min. 32 characters).");
          }
          if (!configService.get('searchguard.cookie.secure')) {
            this.logger.warn("'searchguard.cookie.secure' is set to false, cookies are transmitted over unsecure HTTP connection. Consider using HTTPS and set this key to 'true'");
          }
        } catch (error) {
          this.logger.error(`An error occurred registering server plugins: ${error}`);
          throw error;
        }
      }

      // @todo TEST
      if (configService.get('searchguard.xff.enabled')) {
        require('./xff/xff')(core);
        this.logger.info('Search Guard XFF enabled.');
      }
      if (configService.get('searchguard.configuration.enabled')) {
        (0, _routes3.defineConfigurationRoutes)({
          searchGuardConfigurationBackend,
          logger: this.logger,
          kibanaCore: core
        });
        this.logger.info('Routes for Search Guard configuration GUI registered. This is an Enterprise feature.');
      } else {
        this.logger.warn('Search Guard configuration GUI disabled');
      }
      (0, _routes2.defineSystemRoutes)({
        searchGuardBackend,
        logger: this.logger,
        kibanaCore: core
      });
      this.logger.info('Search Guard system routes registered.');
      return {
        authManager,
        sessionStorageFactory,
        kerberos
      };
    } catch (error) {
      this.logger.error(error);
      throw error;
    }
  }
}
exports.SearchGuard = SearchGuard;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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